<?php
require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/../config/Database.php';

$db = Database::getInstance()->getConnection();

// Get statistics
$usersCount = $db->query("SELECT COUNT(*) as count FROM users")->fetch()['count'];
$activeUsers = $db->query("SELECT COUNT(*) as count FROM users WHERE last_activity > DATE_SUB(NOW(), INTERVAL 30 DAY)")->fetch()['count'];
$messagesCount = $db->query("SELECT COUNT(*) as count FROM messages")->fetch()['count'];
$todayMessages = $db->query("SELECT COUNT(*) as count FROM messages WHERE DATE(created_at) = CURDATE()")->fetch()['count'];

// Get recent users
$recentUsers = $db->query("
    SELECT user_id, username, first_name, last_name, last_activity 
    FROM users 
    ORDER BY last_activity DESC 
    LIMIT 5
")->fetchAll();

// Get message stats
$messageStats = $db->query("
    SELECT 
        DATE(created_at) as date,
        COUNT(*) as count,
        SUM(CASE WHEN direction = 'incoming' THEN 1 ELSE 0 END) as incoming,
        SUM(CASE WHEN direction = 'outgoing' THEN 1 ELSE 0 END) as outgoing
    FROM messages
    WHERE created_at > DATE_SUB(NOW(), INTERVAL 7 DAY)
    GROUP BY DATE(created_at)
    ORDER BY date ASC
")->fetchAll();

// Prepare data for chart
$chartLabels = [];
$chartIncoming = [];
$chartOutgoing = [];

foreach ($messageStats as $stat) {
    $chartLabels[] = date('D', strtotime($stat['date']));
    $chartIncoming[] = $stat['incoming'];
    $chartOutgoing[] = $stat['outgoing'];
}

require_once __DIR__ . '/includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <?php require_once __DIR__ . '/includes/sidebar.php'; ?>
        
        <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2">Dashboard</h1>
                <div class="btn-toolbar mb-2 mb-md-0">
                    <div class="btn-group me-2">
                        <button type="button" class="btn btn-sm btn-outline-secondary">Share</button>
                        <button type="button" class="btn btn-sm btn-outline-secondary">Export</button>
                    </div>
                    <button type="button" class="btn btn-sm btn-outline-secondary dropdown-toggle">
                        <span data-feather="calendar"></span>
                        This week
                    </button>
                </div>
            </div>
            
            <!-- Stats Cards -->
            <div class="row mb-4">
                <div class="col-md-3">
                    <div class="card text-white bg-primary mb-3">
                        <div class="card-body">
                            <h5 class="card-title">Total Pengguna</h5>
                            <h2 class="card-text"><?= number_format($usersCount) ?></h2>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card text-white bg-success mb-3">
                        <div class="card-body">
                            <h5 class="card-title">Pengguna Aktif</h5>
                            <h2 class="card-text"><?= number_format($activeUsers) ?></h2>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card text-white bg-info mb-3">
                        <div class="card-body">
                            <h5 class="card-title">Total Pesan</h5>
                            <h2 class="card-text"><?= number_format($messagesCount) ?></h2>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card text-white bg-warning mb-3">
                        <div class="card-body">
                            <h5 class="card-title">Pesan Hari Ini</h5>
                            <h2 class="card-text"><?= number_format($todayMessages) ?></h2>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Charts -->
            <div class="row">
                <div class="col-md-8">
                    <div class="card mb-4">
                        <div class="card-header">
                            <h5>Aktivitas Pesan (7 Hari Terakhir)</h5>
                        </div>
                        <div class="card-body">
                            <canvas id="messageChart" height="250"></canvas>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card mb-4">
                        <div class="card-header">
                            <h5>Pengguna Terbaru</h5>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-sm">
                                    <thead>
                                        <tr>
                                            <th>Nama</th>
                                            <th>Terakhir Aktif</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($recentUsers as $user): ?>
                                            <tr>
                                                <td>
                                                    <?= htmlspecialchars($user['first_name'] . ' ' . htmlspecialchars($user['last_name'])) ?>
                                                    <?php if ($user['username']): ?>
                                                        <br><small class="text-muted">@<?= htmlspecialchars($user['username']) ?></small>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?= date('d M Y H:i', strtotime($user['last_activity'])) ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Recent Messages -->
            <div class="card">
                <div class="card-header">
                    <h5>Pesan Terakhir</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Chat</th>
                                    <th>Pesan</th>
                                    <th>Tipe</th>
                                    <th>Waktu</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                $recentMessages = $db->query("
                                    SELECT m.*, u.first_name, u.last_name, u.username 
                                    FROM messages m
                                    LEFT JOIN users u ON m.chat_id = u.user_id
                                    ORDER BY m.created_at DESC 
                                    LIMIT 10
                                ")->fetchAll();
                                
                                foreach ($recentMessages as $message): ?>
                                    <tr>
                                        <td><?= $message['id'] ?></td>
                                        <td>
                                            <?= htmlspecialchars($message['first_name'] . ' ' . htmlspecialchars($message['last_name'])) ?>
                                            <?php if ($message['username']): ?>
                                                <br><small class="text-muted">@<?= htmlspecialchars($message['username']) ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= htmlspecialchars(substr($message['message_text'], 0, 50)) ?>...</td>
                                        <td><?= $message['message_type'] ?></td>
                                        <td><?= date('d M H:i', strtotime($message['created_at'])) ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
    // Message Chart
    const ctx = document.getElementById('messageChart').getContext('2d');
    const messageChart = new Chart(ctx, {
        type: 'line',
        data: {
            labels: <?= json_encode($chartLabels) ?>,
            datasets: [
                {
                    label: 'Pesan Masuk',
                    data: <?= json_encode($chartIncoming) ?>,
                    backgroundColor: 'rgba(54, 162, 235, 0.2)',
                    borderColor: 'rgba(54, 162, 235, 1)',
                    borderWidth: 2,
                    tension: 0.1
                },
                {
                    label: 'Pesan Keluar',
                    data: <?= json_encode($chartOutgoing) ?>,
                    backgroundColor: 'rgba(75, 192, 192, 0.2)',
                    borderColor: 'rgba(75, 192, 192, 1)',
                    borderWidth: 2,
                    tension: 0.1
                }
            ]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    position: 'top',
                },
                tooltip: {
                    mode: 'index',
                    intersect: false,
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        stepSize: 1
                    }
                }
            }
        }
    });
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>