<?php
class Database {
    private static $instance = null;
    private $connection;
    
    private function __construct() {
        $host = Config::get('DB_HOST');
        $dbname = Config::get('DB_NAME');
        $username = Config::get('DB_USER');
        $password = Config::get('DB_PASS');
        $charset = Config::get('DB_CHARSET', 'utf8mb4');
        
        $dsn = "mysql:host=$host;dbname=$dbname;charset=$charset";
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ];
        
        try {
            $this->connection = new PDO($dsn, $username, $password, $options);
        } catch (PDOException $e) {
            throw new Exception("Database connection failed: " . $e->getMessage());
        }
    }
    
    public static function getInstance() {
        if (!self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        return $this->connection;
    }
    
    // CRUD Methods
    public function insert($table, $data) {
        $columns = implode(', ', array_keys($data));
        $placeholders = ':' . implode(', :', array_keys($data));
        
        $sql = "INSERT INTO $table ($columns) VALUES ($placeholders)";
        $stmt = $this->connection->prepare($sql);
        
        foreach ($data as $key => $value) {
            $stmt->bindValue(":$key", $value);
        }
        
        $stmt->execute();
        return $this->connection->lastInsertId();
    }
    
    public function select($table, $conditions = [], $limit = null, $offset = null) {
        $sql = "SELECT * FROM $table";
        $params = [];
        
        if (!empty($conditions)) {
            $where = [];
            foreach ($conditions as $key => $value) {
                $where[] = "$key = :$key";
                $params[":$key"] = $value;
            }
            $sql .= " WHERE " . implode(' AND ', $where);
        }
        
        if ($limit !== null) {
            $sql .= " LIMIT :limit";
            $params[':limit'] = (int)$limit;
            
            if ($offset !== null) {
                $sql .= " OFFSET :offset";
                $params[':offset'] = (int)$offset;
            }
        }
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }
    
    public function update($table, $data, $conditions) {
        $set = [];
        $params = [];
        
        foreach ($data as $key => $value) {
            $set[] = "$key = :set_$key";
            $params[":set_$key"] = $value;
        }
        
        $where = [];
        foreach ($conditions as $key => $value) {
            $where[] = "$key = :cond_$key";
            $params[":cond_$key"] = $value;
        }
        
        $sql = "UPDATE $table SET " . implode(', ', $set) . " WHERE " . implode(' AND ', $where);
        $stmt = $this->connection->prepare($sql);
        return $stmt->execute($params);
    }
    
    public function delete($table, $conditions) {
        $where = [];
        $params = [];
        
        foreach ($conditions as $key => $value) {
            $where[] = "$key = :$key";
            $params[":$key"] = $value;
        }
        
        $sql = "DELETE FROM $table WHERE " . implode(' AND ', $where);
        $stmt = $this->connection->prepare($sql);
        return $stmt->execute($params);
    }
    
    // Khusus untuk statistik bot
    public function getMessageStats($period = 'day') {
        $sql = "SELECT 
                    DATE(created_at) as date, 
                    COUNT(*) as message_count,
                    SUM(CASE WHEN message_type = 'text' THEN 1 ELSE 0 END) as text_count,
                    SUM(CASE WHEN message_type = 'photo' THEN 1 ELSE 0 END) as photo_count,
                    SUM(CASE WHEN message_type = 'document' THEN 1 ELSE 0 END) as document_count
                FROM messages";
        
        switch ($period) {
            case 'hour':
                $sql = str_replace('DATE(created_at)', 'DATE_FORMAT(created_at, "%Y-%m-%d %H:00")', $sql);
                $sql .= " GROUP BY DATE_FORMAT(created_at, '%Y-%m-%d %H:00')";
                break;
            case 'week':
                $sql .= " GROUP BY YEARWEEK(created_at, 1)";
                break;
            case 'month':
                $sql .= " GROUP BY YEAR(created_at), MONTH(created_at)";
                break;
            default: // day
                $sql .= " GROUP BY DATE(created_at)";
        }
        
        $sql .= " ORDER BY date DESC LIMIT 30";
        
        $stmt = $this->connection->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    // Backup database
    public function backup($filePath) {
        $tables = $this->connection->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
        $output = '';
        
        foreach ($tables as $table) {
            $output .= "DROP TABLE IF EXISTS `$table`;\n";
            $createTable = $this->connection->query("SHOW CREATE TABLE `$table`")->fetch();
            $output .= $createTable['Create Table'] . ";\n\n";
            
            $rows = $this->connection->query("SELECT * FROM `$table`")->fetchAll(PDO::FETCH_ASSOC);
            if (count($rows) > 0) {
                $columns = array_keys($rows[0]);
                $output .= "INSERT INTO `$table` (`" . implode('`, `', $columns) . "`) VALUES \n";
                
                $values = [];
                foreach ($rows as $row) {
                    $rowValues = array_map(function($value) {
                        return $this->connection->quote($value);
                    }, $row);
                    $values[] = "(" . implode(', ', $rowValues) . ")";
                }
                
                $output .= implode(",\n", $values) . ";\n\n";
            }
        }
        
        return file_put_contents($filePath, $output);
    }
}