<?php
class TelegramBot {
    private $token;
    private $apiUrl = 'https://api.telegram.org/bot';
    private $db;
    
    public function __construct() {
        $this->token = Config::get('TELEGRAM_BOT_TOKEN');
        $this->db = Database::getInstance()->getConnection();
    }
    
    private function sendRequest($method, $params = [], $file = null) {
        $url = $this->apiUrl . $this->token . '/' . $method;
        $ch = curl_init();
        
        if ($file) {
            // Handle file upload
            $params[$file['param']] = new CURLFile($file['path']);
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: multipart/form-data']);
        } else {
            // JSON encode for normal requests
            $params = json_encode($params);
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        }
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $params,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CONNECTTIMEOUT => 5
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            error_log("Telegram API Error: $error | HTTP Code: $httpCode");
            return false;
        }
        
        return json_decode($response, true);
    }
    
    // Webhook Methods
    public function setWebhook($url, $max_connections = 40, $allowed_updates = null) {
        $params = [
            'url' => $url,
            'max_connections' => $max_connections
        ];
        
        if ($allowed_updates) {
            $params['allowed_updates'] = json_encode($allowed_updates);
        }
        
        return $this->sendRequest('setWebhook', $params);
    }
    
    public function deleteWebhook() {
        return $this->sendRequest('deleteWebhook');
    }
    
    public function getWebhookInfo() {
        return $this->sendRequest('getWebhookInfo');
    }
    
    // Message Methods
    public function sendMessage($chatId, $text, $replyMarkup = null, $parseMode = 'HTML', $replyTo = null) {
        $params = [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => $parseMode
        ];
        
        if ($replyMarkup) {
            $params['reply_markup'] = json_encode($replyMarkup);
        }
        
        if ($replyTo) {
            $params['reply_to_message_id'] = $replyTo;
        }
        
        // Log pesan ke database
        $this->logMessage($chatId, $text, 'outgoing', 'text');
        
        return $this->sendRequest('sendMessage', $params);
    }
    
    public function sendPhoto($chatId, $photo, $caption = '', $replyMarkup = null) {
        $params = [
            'chat_id' => $chatId,
            'caption' => $caption
        ];
        
        if (filter_var($photo, FILTER_VALIDATE_URL)) {
            $params['photo'] = $photo;
        } else {
            // Handle file upload
            return $this->sendRequest('sendPhoto', $params, [
                'param' => 'photo',
                'path' => $photo
            ]);
        }
        
        if ($replyMarkup) {
            $params['reply_markup'] = json_encode($replyMarkup);
        }
        
        // Log pesan ke database
        $this->logMessage($chatId, $caption ?: 'Photo sent', 'outgoing', 'photo');
        
        return $this->sendRequest('sendPhoto', $params);
    }
    
    public function sendDocument($chatId, $document, $caption = '', $replyMarkup = null) {
        $params = [
            'chat_id' => $chatId,
            'caption' => $caption
        ];
        
        if (filter_var($document, FILTER_VALIDATE_URL)) {
            $params['document'] = $document;
        } else {
            // Handle file upload
            return $this->sendRequest('sendDocument', $params, [
                'param' => 'document',
                'path' => $document
            ]);
        }
        
        if ($replyMarkup) {
            $params['reply_markup'] = json_encode($replyMarkup);
        }
        
        // Log pesan ke database
        $this->logMessage($chatId, $caption ?: 'Document sent', 'outgoing', 'document');
        
        return $this->sendRequest('sendDocument', $params);
    }
    
    // Edit Message Methods
    public function editMessageText($chatId, $messageId, $text, $replyMarkup = null) {
        $params = [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $text
        ];
        
        if ($replyMarkup) {
            $params['reply_markup'] = json_encode($replyMarkup);
        }
        
        return $this->sendRequest('editMessageText', $params);
    }
    
    // Delete Message
    public function deleteMessage($chatId, $messageId) {
        return $this->sendRequest('deleteMessage', [
            'chat_id' => $chatId,
            'message_id' => $messageId
        ]);
    }
    
    // Chat Actions
    public function sendChatAction($chatId, $action = 'typing') {
        $validActions = [
            'typing', 'upload_photo', 'record_video', 'upload_video',
            'record_audio', 'upload_audio', 'upload_document', 'find_location'
        ];
        
        if (!in_array($action, $validActions)) {
            $action = 'typing';
        }
        
        return $this->sendRequest('sendChatAction', [
            'chat_id' => $chatId,
            'action' => $action
        ]);
    }
    
    // User Management
    public function getChat($chatId) {
        return $this->sendRequest('getChat', [
            'chat_id' => $chatId
        ]);
    }
    
    public function getChatMember($chatId, $userId) {
        return $this->sendRequest('getChatMember', [
            'chat_id' => $chatId,
            'user_id' => $userId
        ]);
    }
    
    // Channel Management
    public function sendToChannel($channelId, $message, $type = 'text', $options = []) {
        switch ($type) {
            case 'photo':
                return $this->sendPhoto($channelId, $options['photo'], $message, $options['reply_markup'] ?? null);
            case 'document':
                return $this->sendDocument($channelId, $options['document'], $message, $options['reply_markup'] ?? null);
            default:
                return $this->sendMessage($channelId, $message, $options['reply_markup'] ?? null);
        }
    }
    
    // Keyboard Methods
    public function buildReplyKeyboard($buttons, $resize = true, $oneTime = false, $selective = false) {
        return [
            'keyboard' => $buttons,
            'resize_keyboard' => $resize,
            'one_time_keyboard' => $oneTime,
            'selective' => $selective
        ];
    }
    
    public function buildInlineKeyboard($buttons) {
        return [
            'inline_keyboard' => $buttons
        ];
    }
    
    public function removeReplyKeyboard($selective = false) {
        return [
            'remove_keyboard' => true,
            'selective' => $selective
        ];
    }
    
    // Logging and Database
    private function logMessage($chatId, $text, $direction, $type) {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO messages 
                (chat_id, message_text, direction, message_type, created_at) 
                VALUES (:chat_id, :message_text, :direction, :type, NOW())
            ");
            
            $stmt->execute([
                ':chat_id' => $chatId,
                ':message_text' => $text,
                ':direction' => $direction,
                ':type' => $type
            ]);
            
            return true;
        } catch (PDOException $e) {
            error_log("Database Error: " . $e->getMessage());
            return false;
        }
    }
    
    public function getUser($userId) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE user_id = :user_id");
        $stmt->execute([':user_id' => $userId]);
        return $stmt->fetch();
    }
    
    public function updateUser($userId, $userData) {
        $fields = [];
        $params = [':user_id' => $userId];
        
        foreach ($userData as $key => $value) {
            $fields[] = "$key = :$key";
            $params[":$key"] = $value;
        }
        
        $stmt = $this->db->prepare("
            UPDATE users SET " . implode(', ', $fields) . " 
            WHERE user_id = :user_id
        ");
        
        return $stmt->execute($params);
    }
    
    // Broadcast to all users
    public function broadcastMessage($text, $options = []) {
        $users = $this->db->query("SELECT user_id FROM users WHERE is_active = 1")->fetchAll(PDO::FETCH_COLUMN);
        $success = 0;
        $failures = 0;
        
        foreach ($users as $userId) {
            try {
                $this->sendMessage($userId, $text, $options['reply_markup'] ?? null);
                $success++;
            } catch (Exception $e) {
                error_log("Failed to send to $userId: " . $e->getMessage());
                $failures++;
            }
        }
        
        return [
            'success' => $success,
            'failures' => $failures,
            'total' => count($users)
        ];
    }
}